/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.streaming;

import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.net.Socket;
import java.net.SocketException;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SocketChannel;
import java.util.Collection;
import java.util.Comparator;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.cassandra.concurrent.NamedThreadFactory;
import org.apache.cassandra.io.util.BufferedDataOutputStreamPlus;
import org.apache.cassandra.io.util.DataOutputStreamPlus;
import org.apache.cassandra.io.util.WrappedDataOutputStreamPlus;
import org.apache.cassandra.net.IncomingStreamingConnection;
import org.apache.cassandra.streaming.StreamSession;
import org.apache.cassandra.streaming.messages.StreamInitMessage;
import org.apache.cassandra.streaming.messages.StreamMessage;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionHandler {
    private static final Logger logger = LoggerFactory.getLogger(ConnectionHandler.class);
    private final StreamSession session;
    private IncomingMessageHandler incoming;
    private OutgoingMessageHandler outgoing;

    ConnectionHandler(StreamSession session) {
        this.session = session;
        this.incoming = new IncomingMessageHandler(session);
        this.outgoing = new OutgoingMessageHandler(session);
    }

    public void initiate() throws IOException {
        logger.debug("[Stream #{}] Sending stream init for incoming stream", (Object)this.session.planId());
        Socket incomingSocket = this.session.createConnection();
        this.incoming.start(incomingSocket, 4, true);
        logger.debug("[Stream #{}] Sending stream init for outgoing stream", (Object)this.session.planId());
        Socket outgoingSocket = this.session.createConnection();
        this.outgoing.start(outgoingSocket, 4, true);
    }

    public void initiateOnReceivingSide(IncomingStreamingConnection connection, boolean isForOutgoing, int version) throws IOException {
        if (isForOutgoing) {
            this.outgoing.start(connection, version);
        } else {
            this.incoming.start(connection, version);
        }
    }

    public ListenableFuture<?> close() {
        logger.debug("[Stream #{}] Closing stream connection handler on {}", (Object)this.session.planId(), (Object)this.session.peer);
        ListenableFuture<?> inClosed = this.closeIncoming();
        ListenableFuture<?> outClosed = this.closeOutgoing();
        return Futures.allAsList((ListenableFuture[])new ListenableFuture[]{inClosed, outClosed});
    }

    public ListenableFuture<?> closeOutgoing() {
        return this.outgoing == null ? Futures.immediateFuture(null) : this.outgoing.close();
    }

    public ListenableFuture<?> closeIncoming() {
        return this.incoming == null ? Futures.immediateFuture(null) : this.incoming.close();
    }

    public void sendMessages(Collection<? extends StreamMessage> messages) {
        for (StreamMessage streamMessage : messages) {
            this.sendMessage(streamMessage);
        }
    }

    public void sendMessage(StreamMessage message) {
        if (this.outgoing.isClosed()) {
            throw new RuntimeException("Outgoing stream handler has been closed");
        }
        this.outgoing.enqueue(message);
    }

    public boolean isOutgoingConnected() {
        return this.outgoing != null && !this.outgoing.isClosed();
    }

    static class OutgoingMessageHandler
    extends MessageHandler {
        private final PriorityBlockingQueue<StreamMessage> messageQueue = new PriorityBlockingQueue<StreamMessage>(64, new Comparator<StreamMessage>(){

            @Override
            public int compare(StreamMessage o1, StreamMessage o2) {
                return o2.getPriority() - o1.getPriority();
            }
        });

        OutgoingMessageHandler(StreamSession session) {
            super(session, true);
        }

        @Override
        protected String name() {
            return "STREAM-OUT";
        }

        public void enqueue(StreamMessage message) {
            this.messageQueue.put(message);
        }

        @Override
        public void run() {
            try {
                StreamMessage next;
                DataOutputStreamPlus out = OutgoingMessageHandler.getWriteChannel(this.socket);
                while (!this.isClosed()) {
                    next = this.messageQueue.poll(1L, TimeUnit.SECONDS);
                    if (next == null) continue;
                    logger.debug("[Stream #{}] Sending {}", (Object)this.session.planId(), (Object)next);
                    this.sendMessage(out, next);
                    if (next.type != StreamMessage.Type.SESSION_FAILED) continue;
                    this.close();
                }
                while ((next = this.messageQueue.poll()) != null) {
                    this.sendMessage(out, next);
                }
            }
            catch (InterruptedException e) {
                throw new AssertionError((Object)e);
            }
            catch (Throwable e) {
                this.session.onError(e);
            }
            finally {
                this.signalCloseDone();
            }
        }

        private void sendMessage(DataOutputStreamPlus out, StreamMessage message) {
            try {
                StreamMessage.serialize(message, out, this.protocolVersion, this.session);
                out.flush();
            }
            catch (SocketException e) {
                this.session.onError(e);
                this.close();
            }
            catch (IOException e) {
                this.session.onError(e);
            }
        }
    }

    static class IncomingMessageHandler
    extends MessageHandler {
        IncomingMessageHandler(StreamSession session) {
            super(session, false);
        }

        @Override
        protected String name() {
            return "STREAM-IN";
        }

        @Override
        public void run() {
            try {
                ReadableByteChannel in = IncomingMessageHandler.getReadChannel(this.socket);
                while (!this.isClosed()) {
                    StreamMessage message = StreamMessage.deserialize(in, this.protocolVersion, this.session);
                    logger.debug("[Stream #{}] Received {}", (Object)this.session.planId(), (Object)message);
                    if (message == null) continue;
                    this.session.messageReceived(message);
                }
            }
            catch (Throwable t) {
                JVMStabilityInspector.inspectThrowable(t);
                this.session.onError(t);
            }
            finally {
                this.signalCloseDone();
            }
        }
    }

    static abstract class MessageHandler
    implements Runnable {
        protected final StreamSession session;
        protected int protocolVersion;
        protected Socket socket;
        private final boolean isOutgoingHandler;
        private final AtomicReference<SettableFuture<?>> closeFuture = new AtomicReference();
        private IncomingStreamingConnection incomingConnection;

        protected MessageHandler(StreamSession session, boolean isOutgoingHandler) {
            this.session = session;
            this.isOutgoingHandler = isOutgoingHandler;
        }

        protected abstract String name();

        protected static DataOutputStreamPlus getWriteChannel(Socket socket) throws IOException {
            SocketChannel out = socket.getChannel();
            if (out == null) {
                return new WrappedDataOutputStreamPlus(new BufferedOutputStream(socket.getOutputStream()));
            }
            return new BufferedDataOutputStreamPlus(out);
        }

        protected static ReadableByteChannel getReadChannel(Socket socket) throws IOException {
            return Channels.newChannel(socket.getInputStream());
        }

        public void sendInitMessage() throws IOException {
            StreamInitMessage message = new StreamInitMessage(FBUtilities.getBroadcastAddress(), this.session.sessionIndex(), this.session.planId(), this.session.description(), !this.isOutgoingHandler, this.session.keepSSTableLevel(), this.session.isIncremental());
            ByteBuffer messageBuf = message.createMessage(false, this.protocolVersion);
            DataOutputStreamPlus out = MessageHandler.getWriteChannel(this.socket);
            out.write(messageBuf);
            out.flush();
        }

        public void start(IncomingStreamingConnection connection, int protocolVersion) throws IOException {
            this.incomingConnection = connection;
            this.start(connection.socket, protocolVersion, false);
        }

        public void start(Socket socket, int protocolVersion, boolean initiator) throws IOException {
            this.socket = socket;
            this.protocolVersion = protocolVersion;
            if (initiator) {
                this.sendInitMessage();
            }
            new Thread(NamedThreadFactory.threadLocalDeallocator(this), this.name() + "-" + this.session.peer).start();
        }

        public ListenableFuture<?> close() {
            SettableFuture future = SettableFuture.create();
            return this.closeFuture.compareAndSet(null, future) ? future : (ListenableFuture)this.closeFuture.get();
        }

        public boolean isClosed() {
            return this.closeFuture.get() != null;
        }

        protected void signalCloseDone() {
            if (!this.isClosed()) {
                this.close();
            }
            this.closeFuture.get().set(null);
            if (this.incomingConnection != null) {
                this.incomingConnection.close();
            } else {
                try {
                    this.socket.close();
                }
                catch (IOException e) {
                    logger.debug("Unexpected error while closing streaming connection", (Throwable)e);
                }
            }
        }
    }
}

