/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.AbstractIterator;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Iterator;
import java.util.zip.CRC32;
import javax.crypto.Cipher;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.CommitLogReadHandler;
import org.apache.cassandra.db.commitlog.EncryptedFileSegmentInputStream;
import org.apache.cassandra.io.FSReadError;
import org.apache.cassandra.io.compress.ICompressor;
import org.apache.cassandra.io.util.FileDataInput;
import org.apache.cassandra.io.util.FileSegmentInputStream;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.schema.CompressionParams;
import org.apache.cassandra.security.EncryptionContext;
import org.apache.cassandra.security.EncryptionUtils;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLogSegmentReader
implements Iterable<SyncSegment> {
    private static volatile boolean allowSkipSyncMarkerCrc = CassandraRelevantProperties.COMMITLOG_ALLOW_IGNORE_SYNC_CRC.getBoolean();
    private static final Logger logger = LoggerFactory.getLogger(CommitLogSegmentReader.class);
    private final CommitLogReadHandler handler;
    private final CommitLogDescriptor descriptor;
    private final RandomAccessReader reader;
    private final Segmenter segmenter;
    private final boolean tolerateTruncation;
    protected int end;

    protected CommitLogSegmentReader(CommitLogReadHandler handler, CommitLogDescriptor descriptor, RandomAccessReader reader, boolean tolerateTruncation) {
        this.handler = handler;
        this.descriptor = descriptor;
        this.reader = reader;
        this.tolerateTruncation = tolerateTruncation;
        this.end = (int)reader.getFilePointer();
        this.segmenter = descriptor.getEncryptionContext().isEnabled() ? new EncryptedSegmenter(descriptor, reader) : (descriptor.compression != null ? new CompressedSegmenter(descriptor, reader) : new NoOpSegmenter(reader));
    }

    public static void setAllowSkipSyncMarkerCrc(boolean allow) {
        allowSkipSyncMarkerCrc = allow;
    }

    @Override
    public Iterator<SyncSegment> iterator() {
        return new SegmentIterator();
    }

    private void handleUnrecoverableError(Exception e, boolean permissible) {
        try {
            this.handler.handleUnrecoverableError(new CommitLogReadHandler.CommitLogReadException(e.getMessage(), CommitLogReadHandler.CommitLogReadErrorReason.UNRECOVERABLE_DESCRIPTOR_ERROR, permissible));
        }
        catch (IOException ioe) {
            throw new RuntimeException(ioe);
        }
    }

    private int readSyncMarker(CommitLogDescriptor descriptor, int offset, RandomAccessReader reader) throws IOException {
        if ((long)offset > reader.length() - 8L) {
            return -1;
        }
        reader.seek(offset);
        CRC32 crc = new CRC32();
        FBUtilities.updateChecksumInt(crc, (int)(descriptor.id & 0xFFFFFFFFL));
        FBUtilities.updateChecksumInt(crc, (int)(descriptor.id >>> 32));
        FBUtilities.updateChecksumInt(crc, (int)reader.getPosition());
        int end = reader.readInt();
        long filecrc = (long)reader.readInt() & 0xFFFFFFFFL;
        if (crc.getValue() != filecrc) {
            if (allowSkipSyncMarkerCrc && descriptor.compression == null && !descriptor.getEncryptionContext().isEnabled() && filecrc == 0L && end != 0) {
                logger.warn("Skipping sync marker CRC check at position {} (end={}, calculated crc={}) of commit log {}.Using per-mutation CRC checks to ensure correctness...", new Object[]{offset, end, crc.getValue(), reader.getPath()});
                return end;
            }
            if (end != 0 || filecrc != 0L) {
                String msg = String.format("Encountered bad header at position %d of commit log %s, with invalid CRC. The end of segment marker should be zero.", offset, reader.getPath());
                throw new SegmentReadException(msg, true);
            }
            return -1;
        }
        if (end < offset || (long)end > reader.length()) {
            String msg = String.format("Encountered bad header at position %d of commit log %s, with bad position but valid CRC", offset, reader.getPath());
            throw new SegmentReadException(msg, false);
        }
        return end;
    }

    static class EncryptedSegmenter
    implements Segmenter {
        private final RandomAccessReader reader;
        private final ICompressor compressor;
        private final Cipher cipher;
        private ByteBuffer decryptedBuffer;
        private ByteBuffer uncompressedBuffer;
        private final EncryptedFileSegmentInputStream.ChunkProvider chunkProvider;
        private long currentSegmentEndPosition;
        private long nextLogicalStart;

        public EncryptedSegmenter(CommitLogDescriptor descriptor, RandomAccessReader reader) {
            this(reader, descriptor.getEncryptionContext());
        }

        @VisibleForTesting
        EncryptedSegmenter(RandomAccessReader reader, EncryptionContext encryptionContext) {
            this.reader = reader;
            this.decryptedBuffer = ByteBuffer.allocate(0);
            this.compressor = encryptionContext.getCompressor();
            this.nextLogicalStart = reader.getFilePointer();
            try {
                this.cipher = encryptionContext.getDecryptor();
            }
            catch (IOException ioe) {
                throw new FSReadError((Throwable)ioe, reader.getPath());
            }
            this.chunkProvider = () -> {
                if (reader.getFilePointer() >= this.currentSegmentEndPosition) {
                    return ByteBufferUtil.EMPTY_BYTE_BUFFER;
                }
                try {
                    this.decryptedBuffer = EncryptionUtils.decrypt(reader, this.decryptedBuffer, true, this.cipher);
                    this.uncompressedBuffer = EncryptionUtils.uncompress(this.decryptedBuffer, this.uncompressedBuffer, true, this.compressor);
                    return this.uncompressedBuffer;
                }
                catch (IOException e) {
                    throw new FSReadError((Throwable)e, reader.getPath());
                }
            };
        }

        @Override
        public SyncSegment nextSegment(int startPosition, int nextSectionStartPosition) throws IOException {
            int totalPlainTextLength = this.reader.readInt();
            this.currentSegmentEndPosition = nextSectionStartPosition - 1;
            this.nextLogicalStart += 8L;
            EncryptedFileSegmentInputStream input = new EncryptedFileSegmentInputStream(this.reader.getPath(), this.nextLogicalStart, 0, totalPlainTextLength, this.chunkProvider);
            this.nextLogicalStart += (long)totalPlainTextLength;
            return new SyncSegment(input, startPosition, nextSectionStartPosition, (int)this.nextLogicalStart, this.tolerateSegmentErrors(nextSectionStartPosition, this.reader.length()));
        }
    }

    static class CompressedSegmenter
    implements Segmenter {
        private final ICompressor compressor;
        private final RandomAccessReader reader;
        private byte[] compressedBuffer;
        private byte[] uncompressedBuffer;
        private long nextLogicalStart;

        public CompressedSegmenter(CommitLogDescriptor desc, RandomAccessReader reader) {
            this(CompressionParams.createCompressor(desc.compression), reader);
        }

        public CompressedSegmenter(ICompressor compressor, RandomAccessReader reader) {
            this.compressor = compressor;
            this.reader = reader;
            this.compressedBuffer = new byte[0];
            this.uncompressedBuffer = new byte[0];
            this.nextLogicalStart = reader.getFilePointer();
        }

        @Override
        public SyncSegment nextSegment(int startPosition, int nextSectionStartPosition) throws IOException {
            this.reader.seek(startPosition);
            int uncompressedLength = this.reader.readInt();
            int compressedLength = nextSectionStartPosition - (int)this.reader.getPosition();
            if (compressedLength > this.compressedBuffer.length) {
                this.compressedBuffer = new byte[(int)(1.2 * (double)compressedLength)];
            }
            this.reader.readFully(this.compressedBuffer, 0, compressedLength);
            if (uncompressedLength > this.uncompressedBuffer.length) {
                this.uncompressedBuffer = new byte[(int)(1.2 * (double)uncompressedLength)];
            }
            int count = this.compressor.uncompress(this.compressedBuffer, 0, compressedLength, this.uncompressedBuffer, 0);
            this.nextLogicalStart += 8L;
            FileSegmentInputStream input = new FileSegmentInputStream(ByteBuffer.wrap(this.uncompressedBuffer, 0, count), this.reader.getPath(), this.nextLogicalStart);
            this.nextLogicalStart += (long)uncompressedLength;
            return new SyncSegment(input, startPosition, nextSectionStartPosition, (int)this.nextLogicalStart, this.tolerateSegmentErrors(nextSectionStartPosition, this.reader.length()));
        }
    }

    static class NoOpSegmenter
    implements Segmenter {
        private final RandomAccessReader reader;

        public NoOpSegmenter(RandomAccessReader reader) {
            this.reader = reader;
        }

        @Override
        public SyncSegment nextSegment(int startPosition, int nextSectionStartPosition) {
            this.reader.seek(startPosition);
            return new SyncSegment(this.reader, startPosition, nextSectionStartPosition, nextSectionStartPosition, true);
        }

        @Override
        public boolean tolerateSegmentErrors(int end, long length) {
            return true;
        }
    }

    static interface Segmenter {
        public SyncSegment nextSegment(int var1, int var2) throws IOException;

        default public boolean tolerateSegmentErrors(int segmentEndPosition, long fileLength) {
            return (long)segmentEndPosition >= fileLength || segmentEndPosition < 0;
        }
    }

    public static class SyncSegment {
        public final FileDataInput input;
        public final int fileStartPosition;
        public final int fileEndPosition;
        public final int endPosition;
        public final boolean toleratesErrorsInSection;

        public SyncSegment(FileDataInput input, int fileStartPosition, int fileEndPosition, int endPosition, boolean toleratesErrorsInSection) {
            this.input = input;
            this.fileStartPosition = fileStartPosition;
            this.fileEndPosition = fileEndPosition;
            this.endPosition = endPosition;
            this.toleratesErrorsInSection = toleratesErrorsInSection;
        }
    }

    public static class SegmentReadException
    extends IOException {
        public final boolean invalidCrc;

        public SegmentReadException(String msg, boolean invalidCrc) {
            super(msg);
            this.invalidCrc = invalidCrc;
        }
    }

    protected class SegmentIterator
    extends AbstractIterator<SyncSegment> {
        protected SegmentIterator() {
        }

        @Override
        protected SyncSegment computeNext() {
            while (true) {
                boolean tolerateErrorsInSection;
                int currentStart = CommitLogSegmentReader.this.end;
                try {
                    CommitLogSegmentReader.this.end = CommitLogSegmentReader.this.readSyncMarker(CommitLogSegmentReader.this.descriptor, currentStart, CommitLogSegmentReader.this.reader);
                }
                catch (SegmentReadException e) {
                    CommitLogSegmentReader.this.handleUnrecoverableError(e, !e.invalidCrc && CommitLogSegmentReader.this.tolerateTruncation);
                    CommitLogSegmentReader.this.end = -1;
                }
                catch (IOException e) {
                    tolerateErrorsInSection = CommitLogSegmentReader.this.tolerateTruncation & CommitLogSegmentReader.this.segmenter.tolerateSegmentErrors(CommitLogSegmentReader.this.end, CommitLogSegmentReader.this.reader.length());
                    CommitLogSegmentReader.this.handleUnrecoverableError(e, tolerateErrorsInSection);
                    CommitLogSegmentReader.this.end = -1;
                }
                if (CommitLogSegmentReader.this.end == -1) {
                    return (SyncSegment)this.endOfData();
                }
                if ((long)CommitLogSegmentReader.this.end > CommitLogSegmentReader.this.reader.length()) {
                    CommitLogSegmentReader.this.end = (int)CommitLogSegmentReader.this.reader.length();
                }
                try {
                    return CommitLogSegmentReader.this.segmenter.nextSegment(currentStart + 8, CommitLogSegmentReader.this.end);
                }
                catch (SegmentReadException e) {
                    CommitLogSegmentReader.this.handleUnrecoverableError(e, !e.invalidCrc && CommitLogSegmentReader.this.tolerateTruncation);
                    continue;
                }
                catch (IOException e) {
                    tolerateErrorsInSection = CommitLogSegmentReader.this.tolerateTruncation & CommitLogSegmentReader.this.segmenter.tolerateSegmentErrors(CommitLogSegmentReader.this.end, CommitLogSegmentReader.this.reader.length());
                    CommitLogSegmentReader.this.handleUnrecoverableError(e, tolerateErrorsInSection);
                    continue;
                }
                break;
            }
        }
    }
}

