/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable.format;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableSet;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Function;
import java.util.function.LongPredicate;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.compaction.CompactionController;
import org.apache.cassandra.db.compaction.CompactionInfo;
import org.apache.cassandra.db.compaction.CompactionInterruptedException;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.rows.UnfilteredRowIterator;
import org.apache.cassandra.dht.LocalPartitioner;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.io.sstable.CorruptSSTableException;
import org.apache.cassandra.io.sstable.IVerifier;
import org.apache.cassandra.io.sstable.KeyIterator;
import org.apache.cassandra.io.sstable.KeyReader;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.SSTableIdentityIterator;
import org.apache.cassandra.io.sstable.format.FilterComponent;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.sstable.format.SSTableReaderWithFilter;
import org.apache.cassandra.io.sstable.format.StatsComponent;
import org.apache.cassandra.io.sstable.metadata.MetadataType;
import org.apache.cassandra.io.util.DataIntegrityMetadata;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.IFilter;
import org.apache.cassandra.utils.OutputHandler;
import org.apache.cassandra.utils.TimeUUID;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class SortedTableVerifier<R extends SSTableReaderWithFilter>
implements IVerifier {
    private static final Logger logger = LoggerFactory.getLogger(SortedTableVerifier.class);
    protected final ColumnFamilyStore cfs;
    protected final R sstable;
    protected final ReadWriteLock fileAccessLock;
    protected final RandomAccessReader dataFile;
    protected final VerifyInfo verifyInfo;
    protected final IVerifier.Options options;
    protected final boolean isOffline;
    protected final Function<String, ? extends Collection<Range<Token>>> tokenLookup;
    protected int goodRows;
    protected final OutputHandler outputHandler;

    public SortedTableVerifier(ColumnFamilyStore cfs, R sstable, OutputHandler outputHandler, boolean isOffline, IVerifier.Options options) {
        this.cfs = cfs;
        this.sstable = sstable;
        this.outputHandler = outputHandler;
        this.fileAccessLock = new ReentrantReadWriteLock();
        this.dataFile = isOffline ? ((SSTableReader)sstable).openDataReader() : ((SSTableReader)sstable).openDataReader(CompactionManager.instance.getRateLimiter());
        this.verifyInfo = new VerifyInfo(this.dataFile, (SSTableReader)sstable, this.fileAccessLock.readLock());
        this.options = options;
        this.isOffline = isOffline;
        this.tokenLookup = options.tokenLookup;
    }

    protected void deserializeBloomFilter(SSTableReader sstable) throws IOException {
        try (IFilter filter = FilterComponent.load(sstable.descriptor);){
            if (filter != null) {
                logger.trace("Filter loaded for {}", (Object)sstable);
            }
        }
    }

    @Override
    public CompactionInfo.Holder getVerifyInfo() {
        return this.verifyInfo;
    }

    protected void markAndThrow(Throwable cause) {
        this.markAndThrow(cause, true);
    }

    protected void markAndThrow(Throwable cause, boolean mutateRepaired) {
        if (mutateRepaired && this.options.mutateRepairStatus) {
            try {
                ((SSTableReader)this.sstable).mutateRepairedAndReload(0L, ((SSTableReader)this.sstable).getPendingRepair(), ((SSTableReader)this.sstable).isTransient());
                this.cfs.getTracker().notifySSTableRepairedStatusChanged(Collections.singleton(this.sstable));
            }
            catch (IOException ioe) {
                this.outputHandler.output("Error mutating repairedAt for SSTable %s, as part of markAndThrow", ((SSTableReader)this.sstable).getFilename());
            }
        }
        Exception e = new Exception(String.format("Invalid SSTable %s, please force %srepair", ((SSTableReader)this.sstable).getFilename(), mutateRepaired && this.options.mutateRepairStatus ? "" : "a full "), cause);
        if (this.options.invokeDiskFailurePolicy) {
            throw new CorruptSSTableException((Throwable)e, ((SSTableReader)this.sstable).getFilename());
        }
        throw new RuntimeException(e);
    }

    @Override
    public void verify() {
        this.verifySSTableVersion();
        this.verifySSTableMetadata();
        this.verifyIndex();
        this.verifyBloomFilter();
        if (this.options.checkOwnsTokens && !this.isOffline && !(this.cfs.getPartitioner() instanceof LocalPartitioner) && this.verifyOwnedRanges() == 0) {
            return;
        }
        if (this.options.quick) {
            return;
        }
        if (this.verifyDigest() && !this.options.extendedVerification) {
            return;
        }
        this.verifySSTable();
        this.outputHandler.output("Verify of %s succeeded. All %d rows read successfully", this.sstable, this.goodRows);
    }

    protected void verifyBloomFilter() {
        try {
            this.outputHandler.debug("Deserializing bloom filter for %s", this.sstable);
            this.deserializeBloomFilter((SSTableReader)this.sstable);
        }
        catch (Throwable t2) {
            this.outputHandler.warn(t2);
            this.markAndThrow(t2);
        }
    }

    protected void verifySSTableMetadata() {
        this.outputHandler.output("Deserializing sstable metadata for %s ", this.sstable);
        try {
            StatsComponent statsComponent = StatsComponent.load(((SSTableReaderWithFilter)this.sstable).descriptor, MetadataType.VALIDATION, MetadataType.STATS, MetadataType.HEADER);
            if (statsComponent.validationMetadata() != null && !statsComponent.validationMetadata().partitioner.equals(((SSTable)this.sstable).getPartitioner().getClass().getCanonicalName())) {
                throw new IOException("Partitioner does not match validation metadata");
            }
        }
        catch (Throwable t2) {
            this.outputHandler.warn(t2);
            this.markAndThrow(t2, false);
        }
    }

    protected void verifySSTableVersion() {
        this.outputHandler.output("Verifying %s (%s)", this.sstable, FBUtilities.prettyPrintMemory(this.dataFile.length()));
        if (this.options.checkVersion && !((SSTableReaderWithFilter)this.sstable).descriptor.version.isLatestVersion()) {
            String msg = String.format("%s is not the latest version, run upgradesstables", this.sstable);
            this.outputHandler.output(msg);
            throw new RuntimeException(msg);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    protected int verifyOwnedRanges() {
        List<Object> ownedRanges = Collections.emptyList();
        this.outputHandler.debug("Checking that all tokens are owned by the current node");
        try (KeyIterator iter2 = ((SSTableReader)this.sstable).keyIterator();){
            ownedRanges = Range.normalize(this.tokenLookup.apply(this.cfs.metadata.keyspace));
            if (ownedRanges.isEmpty()) {
                int n = 0;
                return n;
            }
            RangeOwnHelper rangeOwnHelper = new RangeOwnHelper(ownedRanges);
            while (iter2.hasNext()) {
                DecoratedKey key = (DecoratedKey)iter2.next();
                rangeOwnHelper.validate(key);
            }
            return ownedRanges.size();
        }
        catch (Throwable t2) {
            this.outputHandler.warn(t2);
            this.markAndThrow(t2);
        }
        return ownedRanges.size();
    }

    protected boolean verifyDigest() {
        boolean passed = true;
        this.outputHandler.output("Checking computed hash of %s ", this.sstable);
        try {
            DataIntegrityMetadata.FileDigestValidator validator = ((SSTableReader)this.sstable).maybeGetDigestValidator();
            if (validator != null) {
                validator.validate();
            } else {
                this.outputHandler.output("Data digest missing, assuming extended verification of disk values");
                passed = false;
            }
        }
        catch (IOException e) {
            this.outputHandler.warn(e);
            this.markAndThrow(e);
        }
        return passed;
    }

    protected void verifySSTable() {
        this.outputHandler.output("Extended Verify requested, proceeding to inspect values");
        try (VerifyController verifyController = new VerifyController(this.cfs);
             KeyReader indexIterator = ((SSTableReader)this.sstable).keyReader();){
            if (indexIterator.dataPosition() != 0L) {
                this.markAndThrow(new RuntimeException("First row position from index != 0: " + indexIterator.dataPosition()));
            }
            List<Range<Token>> ownedRanges = this.isOffline ? Collections.emptyList() : Range.normalize(this.tokenLookup.apply(this.cfs.metadata().keyspace));
            RangeOwnHelper rangeOwnHelper = new RangeOwnHelper(ownedRanges);
            DecoratedKey prevKey = null;
            while (!this.dataFile.isEOF()) {
                if (this.verifyInfo.isStopRequested()) {
                    throw new CompactionInterruptedException(this.verifyInfo.getCompactionInfo());
                }
                long rowStart = this.dataFile.getFilePointer();
                this.outputHandler.debug("Reading row at %d", rowStart);
                DecoratedKey key = null;
                try {
                    key = ((SSTable)this.sstable).decorateKey(ByteBufferUtil.readWithShortLength(this.dataFile));
                }
                catch (Throwable th) {
                    this.markAndThrow(th);
                }
                if (this.options.checkOwnsTokens && ownedRanges.size() > 0 && !(this.cfs.getPartitioner() instanceof LocalPartitioner)) {
                    try {
                        rangeOwnHelper.validate(key);
                    }
                    catch (Throwable t2) {
                        this.outputHandler.warn(t2, "Key %s in sstable %s not owned by local ranges %s", key, this.sstable, ownedRanges);
                        this.markAndThrow(t2);
                    }
                }
                ByteBuffer currentIndexKey = indexIterator.key();
                long nextRowPositionFromIndex = 0L;
                try {
                    nextRowPositionFromIndex = indexIterator.advance() ? indexIterator.dataPosition() : this.dataFile.length();
                }
                catch (Throwable th) {
                    this.markAndThrow(th);
                }
                long dataStart = this.dataFile.getFilePointer();
                long dataStartFromIndex = currentIndexKey == null ? -1L : rowStart + 2L + (long)currentIndexKey.remaining();
                long dataSize = nextRowPositionFromIndex - dataStartFromIndex;
                String keyName = key == null ? "(unreadable key)" : ByteBufferUtil.bytesToHex(key.getKey());
                this.outputHandler.debug("row %s is %s", keyName, FBUtilities.prettyPrintMemory(dataSize));
                try {
                    if (key == null || dataSize > this.dataFile.length()) {
                        this.markAndThrow(new RuntimeException(String.format("key = %s, dataSize=%d, dataFile.length() = %d", key, dataSize, this.dataFile.length())));
                    }
                    try (SSTableIdentityIterator iterator = SSTableIdentityIterator.create(this.sstable, this.dataFile, key);){
                        this.verifyPartition(key, iterator);
                    }
                    if (prevKey != null && prevKey.compareTo(key) > 0 || !key.getKey().equals(currentIndexKey) || dataStart != dataStartFromIndex) {
                        this.markAndThrow(new RuntimeException("Key out of order: previous = " + prevKey + " : current = " + key));
                    }
                    ++this.goodRows;
                    prevKey = key;
                    this.outputHandler.debug("Row %s at %s valid, moving to next row at %s ", this.goodRows, rowStart, nextRowPositionFromIndex);
                    this.dataFile.seek(nextRowPositionFromIndex);
                }
                catch (Throwable th) {
                    this.markAndThrow(th);
                }
            }
        }
        catch (Throwable t3) {
            Throwables.throwIfUnchecked(t3);
            throw new RuntimeException(t3);
        }
    }

    protected abstract void verifyPartition(DecoratedKey var1, UnfilteredRowIterator var2);

    protected void verifyIndex() {
        try {
            this.outputHandler.debug("Deserializing index for %s", this.sstable);
            this.deserializeIndex((SSTableReader)this.sstable);
        }
        catch (Throwable t2) {
            this.outputHandler.warn(t2);
            this.markAndThrow(t2);
        }
    }

    private void deserializeIndex(SSTableReader sstable) throws IOException {
        try (KeyReader it = sstable.keyReader();){
            ByteBuffer last = it.key();
            while (it.advance()) {
                last = it.key();
            }
            if (!Objects.equals(last, sstable.getLast().getKey())) {
                throw new CorruptSSTableException((Throwable)new IOException("Failed to read partition index"), it.toString());
            }
        }
    }

    @Override
    public void close() {
        this.fileAccessLock.writeLock().lock();
        try {
            FileUtils.closeQuietly(this.dataFile);
        }
        finally {
            this.fileAccessLock.writeLock().unlock();
        }
    }

    protected static class VerifyController
    extends CompactionController {
        public VerifyController(ColumnFamilyStore cfs) {
            super(cfs, Integer.MAX_VALUE);
        }

        @Override
        public LongPredicate getPurgeEvaluator(DecoratedKey key) {
            return time -> false;
        }
    }

    protected static class VerifyInfo
    extends CompactionInfo.Holder {
        private final RandomAccessReader dataFile;
        private final SSTableReader sstable;
        private final TimeUUID verificationCompactionId;
        private final Lock fileReadLock;

        public VerifyInfo(RandomAccessReader dataFile, SSTableReader sstable, Lock fileReadLock) {
            this.dataFile = dataFile;
            this.sstable = sstable;
            this.fileReadLock = fileReadLock;
            this.verificationCompactionId = TimeUUID.Generator.nextTimeUUID();
        }

        @Override
        public CompactionInfo getCompactionInfo() {
            this.fileReadLock.lock();
            try {
                CompactionInfo compactionInfo = new CompactionInfo(this.sstable.metadata(), OperationType.VERIFY, this.dataFile.getFilePointer(), this.dataFile.length(), this.verificationCompactionId, ImmutableSet.of(this.sstable));
                return compactionInfo;
            }
            catch (Exception e) {
                throw new RuntimeException();
            }
            finally {
                this.fileReadLock.unlock();
            }
        }

        @Override
        public boolean isGlobal() {
            return false;
        }
    }

    @VisibleForTesting
    public static class RangeOwnHelper {
        private final List<Range<Token>> normalizedRanges;
        private int rangeIndex = 0;
        private DecoratedKey lastKey;

        public RangeOwnHelper(List<Range<Token>> normalizedRanges) {
            this.normalizedRanges = normalizedRanges;
            Range.assertNormalized(normalizedRanges);
        }

        public void validate(DecoratedKey key) {
            if (!this.check(key)) {
                throw new RuntimeException("Key " + key + " is not contained in the given ranges");
            }
        }

        public boolean check(DecoratedKey key) {
            assert (this.lastKey == null || key.compareTo(this.lastKey) > 0);
            this.lastKey = key;
            if (this.normalizedRanges.isEmpty()) {
                return true;
            }
            if (this.rangeIndex > this.normalizedRanges.size() - 1) {
                throw new IllegalStateException("RangeOwnHelper can only be used to find the first out-of-range-token");
            }
            while (!this.normalizedRanges.get(this.rangeIndex).contains(key.getToken())) {
                ++this.rangeIndex;
                if (this.rangeIndex <= this.normalizedRanges.size() - 1) continue;
                return false;
            }
            return true;
        }
    }
}

