/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.streaming;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.InetSocketAddress;
import java.util.Collections;
import java.util.EnumMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import javax.annotation.concurrent.GuardedBy;
import org.apache.cassandra.cache.IMeasurableMemory;
import org.apache.cassandra.db.virtual.SimpleDataSet;
import org.apache.cassandra.streaming.ProgressInfo;
import org.apache.cassandra.streaming.SessionInfo;
import org.apache.cassandra.streaming.StreamEvent;
import org.apache.cassandra.streaming.StreamEventHandler;
import org.apache.cassandra.streaming.StreamManager;
import org.apache.cassandra.streaming.StreamOperation;
import org.apache.cassandra.streaming.StreamResultFuture;
import org.apache.cassandra.streaming.StreamSession;
import org.apache.cassandra.streaming.StreamState;
import org.apache.cassandra.tools.nodetool.formatter.TableBuilder;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.ObjectSizes;
import org.apache.cassandra.utils.TimeUUID;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StreamingState
implements StreamEventHandler,
IMeasurableMemory {
    private static final Logger logger = LoggerFactory.getLogger(StreamingState.class);
    public static final long EMPTY = ObjectSizes.measureDeep(new StreamingState(TimeUUID.Generator.nextTimeUUID(), StreamOperation.OTHER, false));
    private final long createdAtMillis = Clock.Global.currentTimeMillis();
    private final TimeUUID id;
    private final boolean follower;
    private final StreamOperation operation;
    private final Set<InetSocketAddress> peers = Collections.newSetFromMap(new ConcurrentHashMap());
    @GuardedBy(value="this")
    private final Sessions sessions = new Sessions();
    private Status status;
    private String completeMessage = null;
    private final long[] stateTimesNanos;
    private volatile long lastUpdatedAtNanos;
    public final Phase phase = new Phase();

    @Override
    public long unsharedHeapSize() {
        long costOfPeers = (long)this.peers().size() * (ObjectSizes.IPV6_SOCKET_ADDRESS_SIZE + 48L);
        long costOfCompleteMessage = ObjectSizes.sizeOf(this.completeMessage());
        return costOfPeers + costOfCompleteMessage + EMPTY;
    }

    public StreamingState(StreamResultFuture result) {
        this(result.planId, result.streamOperation, result.getCoordinator().isFollower());
    }

    private StreamingState(TimeUUID planId, StreamOperation streamOperation, boolean follower) {
        this.id = planId;
        this.operation = streamOperation;
        this.follower = follower;
        this.stateTimesNanos = new long[Status.values().length];
        this.updateState(Status.INIT);
    }

    public TimeUUID id() {
        return this.id;
    }

    public boolean follower() {
        return this.follower;
    }

    public StreamOperation operation() {
        return this.operation;
    }

    public Set<InetSocketAddress> peers() {
        return this.peers;
    }

    public String completeMessage() {
        return this.completeMessage;
    }

    public Status status() {
        return this.status;
    }

    public Sessions sessions() {
        return this.sessions;
    }

    public boolean isComplete() {
        switch (this.status) {
            case SUCCESS: 
            case FAILURE: {
                return true;
            }
        }
        return false;
    }

    @VisibleForTesting
    public StreamResultFuture future() {
        if (this.follower) {
            return StreamManager.instance.getReceivingStream(this.id);
        }
        return StreamManager.instance.getInitiatorStream(this.id);
    }

    public float progress() {
        switch (this.status) {
            case INIT: {
                return 0.0f;
            }
            case START: {
                return Math.min(0.99f, this.sessions().progress().floatValue());
            }
            case SUCCESS: 
            case FAILURE: {
                return 1.0f;
            }
        }
        throw new AssertionError((Object)("unknown state: " + this.status));
    }

    public EnumMap<Status, Long> stateTimesMillis() {
        EnumMap<Status, Long> map = new EnumMap<Status, Long>(Status.class);
        for (int i = 0; i < this.stateTimesNanos.length; ++i) {
            long nanos = this.stateTimesNanos[i];
            if (nanos == 0L) continue;
            map.put(Status.values()[i], this.nanosToMillis(nanos));
        }
        return map;
    }

    public long durationMillis() {
        long endNanos = this.lastUpdatedAtNanos;
        if (!this.isComplete()) {
            endNanos = Clock.Global.nanoTime();
        }
        return TimeUnit.NANOSECONDS.toMillis(endNanos - this.stateTimesNanos[0]);
    }

    public long lastUpdatedAtMillis() {
        return this.nanosToMillis(this.lastUpdatedAtNanos);
    }

    public long lastUpdatedAtNanos() {
        return this.lastUpdatedAtNanos;
    }

    public String failureCause() {
        if (this.status == Status.FAILURE) {
            return this.completeMessage;
        }
        return null;
    }

    public String successMessage() {
        if (this.status == Status.SUCCESS) {
            return this.completeMessage;
        }
        return null;
    }

    public String toString() {
        TableBuilder table = new TableBuilder();
        table.add("id", this.id.toString());
        table.add("status", this.status().name().toLowerCase());
        table.add("progress", this.progress() * 100.0f + "%");
        table.add("duration_ms", Long.toString(this.durationMillis()));
        table.add("last_updated_ms", Long.toString(this.lastUpdatedAtMillis()));
        table.add("failure_cause", this.failureCause());
        table.add("success_message", this.successMessage());
        for (Map.Entry<Status, Long> e : this.stateTimesMillis().entrySet()) {
            table.add("status_" + e.getKey().name().toLowerCase() + "_ms", e.toString());
        }
        return table.toString();
    }

    @Override
    public synchronized void handleStreamEvent(StreamEvent event) {
        try {
            switch (event.eventType) {
                case STREAM_PREPARED: {
                    this.streamPrepared((StreamEvent.SessionPreparedEvent)event);
                    break;
                }
                case STREAM_COMPLETE: {
                    break;
                }
                case FILE_PROGRESS: {
                    this.streamProgress((StreamEvent.ProgressEvent)event);
                    break;
                }
                default: {
                    logger.warn("Unknown stream event type: {}", (Object)event.eventType);
                    break;
                }
            }
        }
        catch (Throwable t2) {
            logger.warn("Unexpected exception handling stream event", t2);
        }
        this.lastUpdatedAtNanos = Clock.Global.nanoTime();
    }

    private void streamPrepared(StreamEvent.SessionPreparedEvent event) {
        SessionInfo session = event.session;
        this.peers.add(session.peer);
        if (event.prepareDirection != StreamSession.PrepareDirection.ACK) {
            return;
        }
        this.sessions.bytesToReceive += session.getTotalSizeToReceive();
        this.sessions.bytesToSend += session.getTotalSizeToSend();
        this.sessions.filesToReceive += session.getTotalFilesToReceive();
        this.sessions.filesToSend += session.getTotalFilesToSend();
    }

    private void streamProgress(StreamEvent.ProgressEvent event) {
        ProgressInfo info = event.progress;
        if (info.direction == ProgressInfo.Direction.IN) {
            this.sessions.bytesReceived += info.deltaBytes;
            if (info.isCompleted()) {
                ++this.sessions.filesReceived;
            }
        } else {
            this.sessions.bytesSent += info.deltaBytes;
            if (info.isCompleted()) {
                ++this.sessions.filesSent;
            }
        }
    }

    @Override
    public synchronized void onSuccess(@Nullable StreamState state) {
        this.updateState(Status.SUCCESS);
    }

    @Override
    public synchronized void onFailure(Throwable throwable) {
        this.completeMessage = Throwables.getStackTraceAsString(throwable);
        this.updateState(Status.FAILURE);
        StreamManager.instance.addStreamingStateAgain(this);
    }

    private synchronized void updateState(Status state) {
        long now;
        this.status = state;
        this.stateTimesNanos[state.ordinal()] = now = Clock.Global.nanoTime();
        this.lastUpdatedAtNanos = now;
    }

    private long nanosToMillis(long nanos) {
        return this.createdAtMillis + TimeUnit.NANOSECONDS.toMillis(nanos - this.stateTimesNanos[0]);
    }

    public static class Sessions {
        public long bytesToReceive;
        public long bytesReceived;
        public long bytesToSend;
        public long bytesSent;
        public long filesToReceive;
        public long filesReceived;
        public long filesToSend;
        public long filesSent;

        public static String columns() {
            return "  bytes_to_receive bigint, \n  bytes_received bigint, \n  bytes_to_send bigint, \n  bytes_sent bigint, \n  files_to_receive bigint, \n  files_received bigint, \n  files_to_send bigint, \n  files_sent bigint, \n";
        }

        public boolean isEmpty() {
            return this.bytesToReceive == 0L && this.bytesToSend == 0L && this.filesToReceive == 0L && this.filesToSend == 0L;
        }

        public BigDecimal progress() {
            return Sessions.div(this.bytesSent + this.bytesReceived, this.bytesToSend + this.bytesToReceive);
        }

        private static BigDecimal div(long a, long b) {
            if (b == 0L) {
                return BigDecimal.ZERO;
            }
            return BigDecimal.valueOf(a).divide(BigDecimal.valueOf(b), 4, RoundingMode.HALF_UP);
        }

        public void update(SimpleDataSet ds) {
            if (this.isEmpty()) {
                return;
            }
            ds.column("bytes_to_receive", this.bytesToReceive).column("bytes_received", this.bytesReceived).column("bytes_to_send", this.bytesToSend).column("bytes_sent", this.bytesSent).column("files_to_receive", this.filesToReceive).column("files_received", this.filesReceived).column("files_to_send", this.filesToSend).column("files_sent", this.filesSent);
        }
    }

    public class Phase {
        public void start() {
            StreamingState.this.updateState(Status.START);
        }
    }

    public static enum Status {
        INIT,
        START,
        SUCCESS,
        FAILURE;

    }
}

