/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import org.apache.iceberg.ManifestEntry;
import org.apache.iceberg.ManifestFile;
import org.apache.iceberg.ManifestFiles;
import org.apache.iceberg.ManifestReader;
import org.apache.iceberg.PartitionStatisticsFile;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.StatisticsFile;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.catalog.Catalog;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.common.DynClasses;
import org.apache.iceberg.common.DynConstructors;
import org.apache.iceberg.common.DynMethods;
import org.apache.iceberg.exceptions.RuntimeIOException;
import org.apache.iceberg.exceptions.ValidationException;
import org.apache.iceberg.hadoop.Configurable;
import org.apache.iceberg.io.FileIO;
import org.apache.iceberg.io.StorageCredential;
import org.apache.iceberg.io.SupportsBulkOperations;
import org.apache.iceberg.io.SupportsStorageCredentials;
import org.apache.iceberg.metrics.LoggingMetricsReporter;
import org.apache.iceberg.metrics.MetricsReporter;
import org.apache.iceberg.relocated.com.google.common.base.Joiner;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableList;
import org.apache.iceberg.relocated.com.google.common.collect.Iterables;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.relocated.com.google.common.collect.MapMaker;
import org.apache.iceberg.relocated.com.google.common.collect.Sets;
import org.apache.iceberg.util.PropertyUtil;
import org.apache.iceberg.util.Tasks;
import org.apache.iceberg.util.ThreadPools;
import org.apache.iceberg.view.ViewMetadata;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CatalogUtil {
    private static final Logger LOG = LoggerFactory.getLogger(CatalogUtil.class);
    public static final String ICEBERG_CATALOG_TYPE = "type";
    public static final String ICEBERG_CATALOG_TYPE_HADOOP = "hadoop";
    public static final String ICEBERG_CATALOG_TYPE_HIVE = "hive";
    public static final String ICEBERG_CATALOG_TYPE_REST = "rest";
    public static final String ICEBERG_CATALOG_TYPE_GLUE = "glue";
    public static final String ICEBERG_CATALOG_TYPE_NESSIE = "nessie";
    public static final String ICEBERG_CATALOG_TYPE_JDBC = "jdbc";
    public static final String ICEBERG_CATALOG_HADOOP = "org.apache.iceberg.hadoop.HadoopCatalog";
    public static final String ICEBERG_CATALOG_HIVE = "org.apache.iceberg.hive.HiveCatalog";
    public static final String ICEBERG_CATALOG_REST = "org.apache.iceberg.rest.RESTCatalog";
    public static final String ICEBERG_CATALOG_GLUE = "org.apache.iceberg.aws.glue.GlueCatalog";
    public static final String ICEBERG_CATALOG_NESSIE = "org.apache.iceberg.nessie.NessieCatalog";
    public static final String ICEBERG_CATALOG_JDBC = "org.apache.iceberg.jdbc.JdbcCatalog";

    private CatalogUtil() {
    }

    public static void dropTableData(FileIO io, TableMetadata metadata) {
        HashSet manifestListsToDelete = Sets.newHashSet();
        HashSet manifestsToDelete = Sets.newHashSet();
        for (Snapshot snapshot : metadata.snapshots()) {
            Iterables.addAll((Collection)manifestsToDelete, (Iterable)snapshot.allManifests(io));
            if (snapshot.manifestListLocation() == null) continue;
            manifestListsToDelete.add(snapshot.manifestListLocation());
        }
        LOG.info("Manifests to delete: {}", (Object)Joiner.on((String)", ").join((Iterable)manifestsToDelete));
        boolean gcEnabled = PropertyUtil.propertyAsBoolean(metadata.properties(), "gc.enabled", true);
        if (gcEnabled) {
            CatalogUtil.deleteFiles(io, manifestsToDelete);
        }
        CatalogUtil.deleteFiles(io, Iterables.transform((Iterable)manifestsToDelete, ManifestFile::path), "manifest", true);
        CatalogUtil.deleteFiles(io, manifestListsToDelete, "manifest list", true);
        CatalogUtil.deleteFiles(io, Iterables.transform(metadata.previousFiles(), TableMetadata.MetadataLogEntry::file), "previous metadata", true);
        CatalogUtil.deleteFiles(io, Iterables.transform(metadata.statisticsFiles(), StatisticsFile::path), "statistics", true);
        CatalogUtil.deleteFiles(io, Iterables.transform(metadata.partitionStatisticsFiles(), PartitionStatisticsFile::path), "partition statistics", true);
        CatalogUtil.deleteFile(io, metadata.metadataFileLocation(), "metadata");
    }

    public static void dropViewMetadata(FileIO io, ViewMetadata metadata) {
        boolean gcEnabled = PropertyUtil.propertyAsBoolean(metadata.properties(), "gc.enabled", true);
        if (gcEnabled) {
            CatalogUtil.deleteFile(io, metadata.metadataFileLocation(), "metadata");
        }
    }

    private static void deleteFiles(FileIO io, Set<ManifestFile> allManifests) {
        ConcurrentMap deletedFiles = new MapMaker().concurrencyLevel(ThreadPools.WORKER_THREAD_POOL_SIZE).weakKeys().makeMap();
        Tasks.foreach(allManifests).noRetry().suppressFailureWhenFinished().executeWith(ThreadPools.getWorkerPool()).onFailure((item, exc) -> LOG.warn("Failed to get deleted files: this may cause orphaned data files", (Throwable)exc)).run(manifest -> {
            try (ManifestReader<?> reader = ManifestFiles.open(manifest, io);){
                ArrayList pathsToDelete = Lists.newArrayList();
                for (ManifestEntry entry : reader.entries()) {
                    String path = entry.file().location().intern();
                    Boolean alreadyDeleted = deletedFiles.putIfAbsent(path, true);
                    if (alreadyDeleted != null && alreadyDeleted.booleanValue()) continue;
                    pathsToDelete.add(path);
                }
                String type = reader.isDeleteManifestReader() ? "delete" : "data";
                CatalogUtil.deleteFiles(io, pathsToDelete, type, false);
            }
            catch (IOException e) {
                throw new RuntimeIOException(e, "Failed to read manifest file: %s", new Object[]{manifest.path()});
            }
        });
    }

    public static void deleteFiles(FileIO io, Iterable<String> files, String type, boolean concurrent) {
        if (io instanceof SupportsBulkOperations) {
            try {
                SupportsBulkOperations bulkIO = (SupportsBulkOperations)io;
                bulkIO.deleteFiles(files);
            }
            catch (RuntimeException e) {
                LOG.warn("Failed to bulk delete {} files", (Object)type, (Object)e);
            }
        } else if (concurrent) {
            CatalogUtil.deleteFiles(io, files, type);
        } else {
            files.forEach(file -> CatalogUtil.deleteFile(io, file, type));
        }
    }

    private static void deleteFiles(FileIO io, Iterable<String> files, String type) {
        Tasks.foreach(files).executeWith(ThreadPools.getWorkerPool()).noRetry().suppressFailureWhenFinished().onFailure((file, exc) -> LOG.warn("Failed to delete {} file {}", new Object[]{type, file, exc})).run(arg_0 -> ((FileIO)io).deleteFile(arg_0));
    }

    private static void deleteFile(FileIO io, String file, String type) {
        try {
            io.deleteFile(file);
        }
        catch (RuntimeException e) {
            LOG.warn("Failed to delete {} file {}", new Object[]{type, file, e});
        }
    }

    public static Catalog loadCatalog(String impl, String catalogName, Map<String, String> properties, Object hadoopConf) {
        Catalog catalog;
        DynConstructors.Ctor ctor;
        Preconditions.checkNotNull((Object)impl, (Object)"Cannot initialize custom Catalog, impl class name is null");
        try {
            ctor = DynConstructors.builder(Catalog.class).impl(impl, new Class[0]).buildChecked();
        }
        catch (NoSuchMethodException e) {
            throw new IllegalArgumentException(String.format("Cannot initialize Catalog implementation %s: %s", impl, e.getMessage()), e);
        }
        try {
            catalog = (Catalog)ctor.newInstance(new Object[0]);
        }
        catch (ClassCastException e) {
            throw new IllegalArgumentException(String.format("Cannot initialize Catalog, %s does not implement Catalog.", impl), e);
        }
        CatalogUtil.configureHadoopConf(catalog, hadoopConf);
        catalog.initialize(catalogName, properties);
        return catalog;
    }

    public static Catalog buildIcebergCatalog(String name, Map<String, String> options, Object conf) {
        String catalogImpl = options.get("catalog-impl");
        if (catalogImpl == null) {
            String catalogType = PropertyUtil.propertyAsString(options, ICEBERG_CATALOG_TYPE, ICEBERG_CATALOG_TYPE_HIVE);
            switch (catalogType.toLowerCase(Locale.ENGLISH)) {
                case "hive": {
                    catalogImpl = ICEBERG_CATALOG_HIVE;
                    break;
                }
                case "hadoop": {
                    catalogImpl = ICEBERG_CATALOG_HADOOP;
                    break;
                }
                case "rest": {
                    catalogImpl = ICEBERG_CATALOG_REST;
                    break;
                }
                case "glue": {
                    catalogImpl = ICEBERG_CATALOG_GLUE;
                    break;
                }
                case "nessie": {
                    catalogImpl = ICEBERG_CATALOG_NESSIE;
                    break;
                }
                case "jdbc": {
                    catalogImpl = ICEBERG_CATALOG_JDBC;
                    break;
                }
                default: {
                    throw new UnsupportedOperationException("Unknown catalog type: " + catalogType);
                }
            }
        } else {
            String catalogType = options.get(ICEBERG_CATALOG_TYPE);
            Preconditions.checkArgument((catalogType == null ? 1 : 0) != 0, (String)"Cannot create catalog %s, both type and catalog-impl are set: type=%s, catalog-impl=%s", (Object)name, (Object)catalogType, (Object)catalogImpl);
        }
        return CatalogUtil.loadCatalog(catalogImpl, name, options, conf);
    }

    public static FileIO loadFileIO(String impl, Map<String, String> properties, Object hadoopConf) {
        return CatalogUtil.loadFileIO(impl, properties, hadoopConf, (List<StorageCredential>)ImmutableList.of());
    }

    public static FileIO loadFileIO(String impl, Map<String, String> properties, Object hadoopConf, List<StorageCredential> storageCredentials) {
        FileIO fileIO;
        DynConstructors.Ctor ctor;
        LOG.info("Loading custom FileIO implementation: {}", (Object)impl);
        try {
            ctor = DynConstructors.builder(FileIO.class).loader(CatalogUtil.class.getClassLoader()).impl(impl, new Class[0]).buildChecked();
        }
        catch (NoSuchMethodException e) {
            throw new IllegalArgumentException(String.format("Cannot initialize FileIO implementation %s: %s", impl, e.getMessage()), e);
        }
        try {
            fileIO = (FileIO)ctor.newInstance(new Object[0]);
        }
        catch (ClassCastException e) {
            throw new IllegalArgumentException(String.format("Cannot initialize FileIO, %s does not implement FileIO.", impl), e);
        }
        CatalogUtil.configureHadoopConf(fileIO, hadoopConf);
        if (fileIO instanceof SupportsStorageCredentials) {
            ((SupportsStorageCredentials)fileIO).setCredentials(storageCredentials);
        }
        fileIO.initialize(properties);
        return fileIO;
    }

    public static void configureHadoopConf(Object maybeConfigurable, Object conf) {
        DynMethods.BoundMethod setConf;
        Class configurationClass;
        Class configurableInterface;
        Preconditions.checkArgument((maybeConfigurable != null ? 1 : 0) != 0, (Object)"Cannot configure: null Configurable");
        if (conf == null) {
            return;
        }
        if (maybeConfigurable instanceof Configurable) {
            ((Configurable)maybeConfigurable).setConf(conf);
            return;
        }
        ClassLoader maybeConfigurableLoader = maybeConfigurable.getClass().getClassLoader();
        try {
            configurableInterface = DynClasses.builder().loader(maybeConfigurableLoader).impl("org.apache.hadoop.conf.Configurable").buildChecked();
        }
        catch (ClassNotFoundException e) {
            return;
        }
        if (!configurableInterface.isInstance(maybeConfigurable)) {
            return;
        }
        try {
            configurationClass = DynClasses.builder().loader(maybeConfigurableLoader).impl("org.apache.hadoop.conf.Configuration").buildChecked();
        }
        catch (ClassNotFoundException e) {
            throw new UnsupportedOperationException("Failed to load Configuration after loading Configurable", e);
        }
        ValidationException.check((boolean)configurationClass.isInstance(conf), (String)"%s is not an instance of Configuration from the classloader for %s", (Object[])new Object[]{conf, maybeConfigurable});
        try {
            setConf = DynMethods.builder((String)"setConf").impl(configurableInterface, new Class[]{configurationClass}).buildChecked().bind(maybeConfigurable);
        }
        catch (NoSuchMethodException e) {
            throw new UnsupportedOperationException("Failed to load Configuration.setConf after loading Configurable", e);
        }
        setConf.invoke(new Object[]{conf});
    }

    public static MetricsReporter loadMetricsReporter(Map<String, String> properties) {
        MetricsReporter reporter;
        DynConstructors.Ctor ctor;
        String impl = properties.get("metrics-reporter-impl");
        if (impl == null) {
            return LoggingMetricsReporter.instance();
        }
        LOG.info("Loading custom MetricsReporter implementation: {}", (Object)impl);
        try {
            ctor = DynConstructors.builder(MetricsReporter.class).loader(CatalogUtil.class.getClassLoader()).impl(impl, new Class[0]).buildChecked();
        }
        catch (NoSuchMethodException e) {
            throw new IllegalArgumentException(String.format("Cannot initialize MetricsReporter, missing no-arg constructor: %s", impl), e);
        }
        try {
            reporter = (MetricsReporter)ctor.newInstance(new Object[0]);
        }
        catch (ClassCastException e) {
            throw new IllegalArgumentException(String.format("Cannot initialize MetricsReporter, %s does not implement MetricsReporter.", impl), e);
        }
        reporter.initialize(properties);
        return reporter;
    }

    public static String fullTableName(String catalogName, TableIdentifier identifier) {
        StringBuilder sb = new StringBuilder();
        if (catalogName.contains("/") || catalogName.contains(":")) {
            sb.append(catalogName);
            if (!catalogName.endsWith("/")) {
                sb.append("/");
            }
        } else {
            sb.append(catalogName).append(".");
        }
        for (String level : identifier.namespace().levels()) {
            sb.append(level).append(".");
        }
        sb.append(identifier.name());
        return sb.toString();
    }

    public static void deleteRemovedMetadataFiles(FileIO io, TableMetadata base, TableMetadata metadata) {
        if (base == null) {
            return;
        }
        boolean deleteAfterCommit = metadata.propertyAsBoolean("write.metadata.delete-after-commit.enabled", false);
        if (deleteAfterCommit) {
            HashSet removedPreviousMetadataFiles = Sets.newHashSet(base.previousFiles());
            removedPreviousMetadataFiles.removeAll(metadata.previousFiles());
            if (io instanceof SupportsBulkOperations) {
                ((SupportsBulkOperations)io).deleteFiles(Iterables.transform((Iterable)removedPreviousMetadataFiles, TableMetadata.MetadataLogEntry::file));
            } else {
                Tasks.foreach(removedPreviousMetadataFiles).noRetry().suppressFailureWhenFinished().onFailure((previousMetadataFile, exc) -> LOG.warn("Delete failed for previous metadata file: {}", previousMetadataFile, (Object)exc)).run(previousMetadataFile -> io.deleteFile(previousMetadataFile.file()));
            }
        }
    }
}

