import { ResourceFetcher } from "../fetch/ResourceFetcher";
import { KubernetesConnectionStatus, isKubernetesConnectionValid } from "./KubernetesConnection";
import { DeploymentState, ResourceDataSource } from "../resources/common";
import { CreateSelfSubjectAccessReview, } from "../resources/kubernetes/SelfSubjectAccessReview";
import { GetNamespace } from "../resources/kubernetes/Namespace";
export class KubernetesService {
    constructor(args) {
        this.args = args;
        this.args = args;
        this.fetcher = new ResourceFetcher({ proxyUrl: args.proxyUrl, connection: this.args.connection });
    }
    composeDeploymentUrlFromIngress(ingress) {
        return `${new URL(this.args.connection.host).origin}/${ingress.metadata.name}`;
    }
    extractDeploymentState(args) {
        var _a;
        if (!args.deployment || !args.deployment.status) {
            return DeploymentState.IN_PROGRESS;
        }
        if (!args.deployment.status.replicas) {
            return DeploymentState.DOWN;
        }
        const progressingCondition = (_a = args.deployment.status.conditions) === null || _a === void 0 ? void 0 : _a.find((condition) => condition.type === "Progressing");
        if (!progressingCondition || progressingCondition.status !== "True") {
            return DeploymentState.DOWN;
        }
        if (!args.deployment.status.readyReplicas) {
            return DeploymentState.IN_PROGRESS;
        }
        return DeploymentState.UP;
    }
    async withFetch(callback) {
        if (!isKubernetesConnectionValid(this.args.connection)) {
            throw new Error("The Kubernetes connection is not valid");
        }
        return callback(this.fetcher);
    }
    async isConnectionEstablished(connection, requiredResources = ["deployments", "services", "ingresses"]) {
        try {
            const selfSubjectAccessReviewResourceName = this.newResourceName("kie-tools");
            const testConnectionFetcher = new ResourceFetcher({ connection, proxyUrl: this.args.proxyUrl });
            try {
                await testConnectionFetcher.execute({ target: new GetNamespace({ namespace: connection.namespace }) });
            }
            catch (e) {
                if (e.cause.status === 404) {
                    return KubernetesConnectionStatus.NAMESPACE_NOT_FOUND;
                }
                throw e;
            }
            const permissionsMap = await Promise.all(requiredResources.map(async (resource) => testConnectionFetcher
                .execute({
                target: new CreateSelfSubjectAccessReview({
                    namespace: connection.namespace,
                    resourceName: selfSubjectAccessReviewResourceName,
                    resourceDataSource: ResourceDataSource.TEMPLATE,
                    createdBy: "kie-tools",
                    resource,
                }),
            })
                .then((result) => { var _a; return ({ resource, allowed: (_a = result.status) === null || _a === void 0 ? void 0 : _a.allowed }); })));
            if (permissionsMap.some((permission) => !permission.allowed)) {
                return KubernetesConnectionStatus.MISSING_PERMISSIONS;
            }
            return KubernetesConnectionStatus.CONNECTED;
        }
        catch (error) {
            return KubernetesConnectionStatus.ERROR;
        }
    }
    newResourceName(prefix) {
        const randomPart = Math.random().toString(36).substring(2, 9);
        const milliseconds = new Date().getMilliseconds();
        const suffix = `${randomPart}${milliseconds}`;
        return `${prefix}-${suffix}`;
    }
}
//# sourceMappingURL=KubernetesService.js.map