/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Phaser;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSInputStream;
import org.apache.hadoop.hdfs.client.impl.DfsClientConf;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LocatedBlocksRefresher
extends Daemon {
    private static final Logger LOG = LoggerFactory.getLogger(LocatedBlocksRefresher.class);
    private static final String THREAD_PREFIX = "located-block-refresher-";
    private final String name;
    private final long interval;
    private final long jitter;
    private final ExecutorService refreshThreadPool;
    private final Set<DFSInputStream> registeredInputStreams = Collections.newSetFromMap(new WeakHashMap());
    private int runCount;
    private int refreshCount;

    LocatedBlocksRefresher(String name, Configuration conf, DfsClientConf dfsClientConf) {
        this.name = name;
        this.interval = dfsClientConf.getLocatedBlocksRefresherInterval();
        this.jitter = Math.round((double)this.interval * 0.1);
        int rpcThreads = conf.getInt("dfs.client.refresh.read-block-locations.threads", 5);
        final String threadPrefix = name.equals("default") ? THREAD_PREFIX : THREAD_PREFIX + name + "-";
        this.refreshThreadPool = Executors.newFixedThreadPool(rpcThreads, new Daemon.DaemonFactory(){
            private final AtomicInteger threadIndex = new AtomicInteger(0);

            @Override
            public Thread newThread(Runnable r) {
                Thread t = super.newThread(r);
                t.setName(threadPrefix + this.threadIndex.getAndIncrement());
                return t;
            }
        });
        this.setName(threadPrefix + "main");
        LOG.info("Start located block refresher for DFSClient {}.", (Object)this.name);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        while (!Thread.currentThread().isInterrupted()) {
            if (!this.waitForInterval()) {
                return;
            }
            LOG.debug("Running refresh for {} streams", (Object)this.registeredInputStreams.size());
            long start = Time.monotonicNow();
            AtomicInteger neededRefresh = new AtomicInteger(0);
            Phaser phaser = new Phaser(1);
            ConcurrentHashMap addressCache = new ConcurrentHashMap();
            for (DFSInputStream inputStream : this.getInputStreams()) {
                phaser.register();
                this.refreshThreadPool.submit(() -> {
                    try {
                        if (this.isInputStreamTracked(inputStream) && inputStream.refreshBlockLocations(addressCache)) {
                            neededRefresh.incrementAndGet();
                        }
                    }
                    finally {
                        phaser.arriveAndDeregister();
                    }
                });
            }
            phaser.arriveAndAwaitAdvance();
            LocatedBlocksRefresher locatedBlocksRefresher = this;
            synchronized (locatedBlocksRefresher) {
                ++this.runCount;
                this.refreshCount += neededRefresh.get();
            }
            LOG.debug("Finished refreshing {} of {} streams in {}ms", new Object[]{neededRefresh, this.registeredInputStreams.size(), Time.monotonicNow() - start});
        }
    }

    public synchronized int getRunCount() {
        return this.runCount;
    }

    public synchronized int getRefreshCount() {
        return this.refreshCount;
    }

    private boolean waitForInterval() {
        try {
            Thread.sleep(this.interval + ThreadLocalRandom.current().nextLong(-this.jitter, this.jitter));
            return true;
        }
        catch (InterruptedException e) {
            LOG.debug("Interrupted during wait interval", (Throwable)e);
            Thread.currentThread().interrupt();
            return false;
        }
    }

    public void shutdown() {
        if (this.isAlive()) {
            this.interrupt();
            try {
                this.join();
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        this.refreshThreadPool.shutdown();
    }

    private synchronized Collection<DFSInputStream> getInputStreams() {
        return new ArrayList<DFSInputStream>(this.registeredInputStreams);
    }

    public synchronized void addInputStream(DFSInputStream dfsInputStream) {
        LOG.trace("Registering {} for {}", (Object)dfsInputStream, (Object)dfsInputStream.getSrc());
        this.registeredInputStreams.add(dfsInputStream);
    }

    public synchronized void removeInputStream(DFSInputStream dfsInputStream) {
        if (this.isInputStreamTracked(dfsInputStream)) {
            LOG.trace("De-registering {} for {}", (Object)dfsInputStream, (Object)dfsInputStream.getSrc());
            this.registeredInputStreams.remove(dfsInputStream);
        }
    }

    public synchronized boolean isInputStreamTracked(DFSInputStream dfsInputStream) {
        return this.registeredInputStreams.contains(dfsInputStream);
    }

    public long getInterval() {
        return this.interval;
    }
}

